@extends('layouts.app')

@php
	$title = $transaction->type == 'sales_order' ? __('lang_v1.edit_sales_order') : __('sale.edit_sale');
@endphp
@section('title', $title)

@php
    /** ---------- EDIT PERMISSIONS BY STATUS ----------
     * Allow editing only when status is: quotation, draft, proforma
     * If status is final, Admin OR Accounting (or users with `sell.update_final`) can edit
     */

    // Derive the "effective" status label used in UI
    if ($transaction->status == 'draft' && $transaction->is_quotation == 1) {
        $effective_status = 'quotation';
    } elseif ($transaction->status == 'draft' && ($transaction->sub_status ?? null) === 'proforma') {
        $effective_status = 'proforma';
    } else {
        $effective_status = $transaction->status; // e.g., 'final', 'draft', ...
    }

    $status_lc = strtolower($effective_status ?? '');
    $is_final  = ($status_lc === 'final');

    $user = auth()->user();

    // ---- Gather every hint we can about roles/titles/user-type ----
    $role_names = collect();

    // Spatie
    if ($user && method_exists($user, 'getRoleNames')) {
        try { $role_names = collect($user->getRoleNames()); } catch (\Throwable $e) {}
    }

    // Common "roles" relation fallback
    if ($role_names->isEmpty() && $user && isset($user->roles)) {
        try { $role_names = collect($user->roles)->pluck('name'); } catch (\Throwable $e) {}
    }

    // Other single-value hints
    $extra_hints = collect([
        $user->role         ?? null,
        $user->user_type    ?? null,
        $user->designation  ?? null,
        $user->title        ?? null,
        $user->department   ?? null,
    ])->filter();

    // Also match when the actual user identity is literally "Accounting"
    $identity_hints = collect([
        $user->username ?? null,
        $user->name     ?? null,
        $user->email    ?? null,
    ])->filter();

    // Normalize everything to lowercase strings
    $all_labels = $role_names
        ->merge($extra_hints)
        ->merge($identity_hints)
        ->filter()
        ->map(function($v){ return strtolower(trim((string)$v)); })
        ->unique();

    // Admin detection
    $is_admin_role =
        $all_labels->contains(fn($v)=> in_array($v, ['admin','administrator'], true)) ||
        $all_labels->contains(fn($v)=> str_contains($v, 'admin')); // e.g., "sysadmin"

    // Accounting/Finance detection (permissive)
    $is_accounting_role =
        $all_labels->contains(fn($v)=> in_array($v, [
            'accounting','accountant','finance','finance team','finance dept',
            'accounting staff','accounting user','accounting dept','accounting team'
        ], true))
        || $all_labels->contains(fn($v)=> str_contains($v, 'accounting'))
        || $all_labels->contains(fn($v)=> str_contains($v, 'accountant'))
        || $all_labels->contains(fn($v)=> str_contains($v, 'finance'))
        || $all_labels->contains(fn($v)=> preg_match('/\baccounts?\b/', $v) === 1);

    // Permission override also allowed (if your app uses gates)
    $has_perm_update_final = $user && method_exists($user, 'can') ? $user->can('sell.update_final') : false;

    // <<< MAIN RULE: Accounting == Admin for Final override >>>
    $can_override_final = $is_admin_role || $is_accounting_role || $has_perm_update_final;

    $allowed_statuses = ['quotation','draft','proforma'];
    $can_edit = in_array($status_lc, $allowed_statuses, true) || ($is_final && $can_override_final);

    // Lock UI when not allowed to edit
    $lock_edit = !$can_edit;
    // Admin OR Accounting can edit Additional Expenses when Final
    $can_edit_additional_expenses = $is_admin_role || $is_accounting_role;

    // Accounting can edit payment area same as Admin
    $can_edit_payment_by_role = $is_admin_role || $is_accounting_role;

    // Only wrap entire form if truly no edits at all
    $wrap_fieldset_disabled = $lock_edit && !($is_final && ($can_edit_payment_by_role || $can_override_final));

    // Let Accounting see payment widgets like Admin
    $allow_payments_widget =
        ($user && method_exists($user,'can') && $user->can('sell.payments'))
        || $is_admin_role
        || $is_accounting_role;
@endphp

@section('content')
<!-- Content Header (Page header) -->
<section class="content-header">
    <h1 class="tw-text-xl md:tw-text-3xl tw-font-bold tw-text-black">
        {{$title}}
        <small class="tw-text-sm md:tw-text-base tw-text-gray-700 tw-font-semibold">
            (@if($transaction->type == 'sales_order') @lang('restaurant.order_no') @else @lang('sale.invoice_no') @endif:
            <span class="text-success">#{{$transaction->invoice_no}})</span>
        </small>
    </h1>

    @if($lock_edit)
        <div class="alert alert-warning" style="margin-top:10px">
            <i class="fa fa-lock"></i>
            @if($is_final)
                This sale status is <strong>Final</strong>. Only admins can edit other fields. Status itself cannot be changed.
            @else
                Editing is disabled for status: <strong>{{ ucfirst($status_lc) }}</strong>.
            @endif
        </div>
    @elseif($is_final)
        <div class="alert alert-info" style="margin-top:10px">
            <i class="fa fa-unlock"></i>
            Status is <strong>Final</strong>. You can edit other fields, but the status itself is locked.
        </div>
    @endif
</section>

<!-- Main content -->
<section class="content">
<input type="hidden" id="amount_rounding_method" value="{{$pos_settings['amount_rounding_method'] ?? ''}}">
<input type="hidden" id="amount_rounding_method" value="{{$pos_settings['amount_rounding_method'] ?? 'none'}}">
@if(!empty($pos_settings['allow_overselling']))
	<input type="hidden" id="is_overselling_allowed">
@endif
@if(session('business.enable_rp') == 1)
    <input type="hidden" id="reward_point_enabled">
@endif
@php
	$custom_labels = json_decode(session('business.custom_labels'), true);
	$common_settings = session()->get('business.common_settings');
@endphp
<input type="hidden" id="item_addition_method" value="{{$business_details->item_addition_method}}">

	{!! Form::open(['url' => action([\App\Http\Controllers\SellPosController::class, 'update'], ['po' => $transaction->id ]), 'method' => 'put', 'id' => 'edit_sell_form', 'files' => true ]) !!}

    {{-- Lock all inputs if not allowed to edit --}}
    @if($lock_edit)
        <fieldset id="lockable-fields" disabled>
    @endif

	{!! Form::hidden('location_id', $transaction->location_id, ['id' => 'location_id', 'data-receipt_printer_type' => !empty($location_printer_type) ? $location_printer_type : 'browser', 'data-default_payment_accounts' => $transaction->location->default_payment_accounts]); !!}

	@if($transaction->type == 'sales_order')
	 	<input type="hidden" id="sale_type" value="{{$transaction->type}}">
	@endif

	<div class="row">
		<div class="col-md-12 col-sm-12">
			@component('components.widget', ['class' => 'box-solid'])
				@if(!empty($transaction->selling_price_group_id))
					<div class="col-md-4 col-sm-6">
						<div class="form-group">
							<div class="input-group">
								<span class="input-group-addon"><i class="fas fa-money-bill-alt"></i></span>
								{!! Form::hidden('price_group', $transaction->selling_price_group_id, ['id' => 'price_group']) !!}
								{!! Form::text('price_group_text', $transaction->price_group->name, ['class' => 'form-control', 'readonly']); !!}
								<span class="input-group-addon">@show_tooltip(__('lang_v1.price_group_help_text'))</span>
							</div>
						</div>
					</div>
				@endif

				@if(in_array('types_of_service', $enabled_modules) && !empty($transaction->types_of_service))
					<div class="col-md-4 col-sm-6">
						<div class="form-group">
							<div class="input-group">
								<span class="input-group-addon"><i class="fas fa-external-link-square-alt text-primary service_modal_btn"></i></span>
								{!! Form::text('types_of_service_text', $transaction->types_of_service->name, ['class' => 'form-control', 'readonly']); !!}
								{!! Form::hidden('types_of_service_id', $transaction->types_of_service_id, ['id' => 'types_of_service_id']) !!}
								<span class="input-group-addon">@show_tooltip(__('lang_v1.types_of_service_help'))</span>
							</div>
							<small><p class="help-block @if(empty($transaction->selling_price_group_id)) hide @endif" id="price_group_text">@lang('lang_v1.price_group'): <span>@if(!empty($transaction->selling_price_group_id)){{$transaction->price_group->name}}@endif</span></p></small>
						</div>
					</div>
					<div class="modal fade types_of_service_modal" tabindex="-1" role="dialog" aria-labelledby="gridSystemModalLabel">
						@if(!empty($transaction->types_of_service))
						@include('types_of_service.pos_form_modal', ['types_of_service' => $transaction->types_of_service])
						@endif
					</div>
				@endif

				@if(in_array('subscription', $enabled_modules))
					<div class="col-md-4 pull-right col-sm-6">
						<div class="checkbox">
							<label>
								{!! Form::checkbox('is_recurring', 1, $transaction->is_recurring, ['class' => 'input-icheck', 'id' => 'is_recurring']); !!} @lang('lang_v1.subscribe')?
				            </label>
							<button type="button" data-toggle="modal" data-target="#recurringInvoiceModal" class="btn btn-link"><i class="fa fa-external-link"></i></button>
							@show_tooltip(__('lang_v1.recurring_invoice_help'))
						</div>
					</div>
				@endif

				<div class="clearfix"></div>

				<div class="@if(!empty($commission_agent)) col-sm-3 @else col-sm-4 @endif">
					<div class="form-group">
						{!! Form::label('contact_id', __('contact.customer') . ':*') !!}
						<div class="input-group">
							<span class="input-group-addon"><i class="fa fa-user"></i></span>
							<input type="hidden" id="default_customer_id" value="{{ $transaction->contact->id }}">
							<input type="hidden" id="default_customer_name" value="{{ $transaction->contact->name }}">
							{!! Form::select('contact_id', [], null, ['class' => 'form-control mousetrap', 'id' => 'customer_id', 'placeholder' => 'Enter Customer name / phone', 'required']); !!}
							<span class="input-group-btn">
								<button type="button" class="btn btn-default bg-white btn-flat add_new_customer" data-name=""><i class="fa fa-plus-circle text-primary fa-lg"></i></button>
							</span>
						</div>
						<small class="text-danger @if(empty($customer_due)) hide @endif contact_due_text"><strong>@lang('account.customer_due'):</strong> <span>{{$customer_due ?? ''}}</span></small>
					</div>
					<small>
						<strong>@lang('lang_v1.billing_address'):</strong>
						<div id="billing_address_div">{!! $transaction->contact->contact_address ?? '' !!}</div>
						<br>
						<strong>@lang('lang_v1.shipping_address'):</strong>
						<div id="shipping_address_div">
							{!! $transaction->contact->supplier_business_name ?? '' !!}, <br>
							{!! $transaction->contact->name ?? '' !!}, <br>
							{!! $transaction->contact->shipping_address ?? '' !!}
						</div>
					</small>
				</div>

				<div class="col-md-3">
		          <div class="form-group">
		            <div class="multi-input">
		            	@php $is_pay_term_required = !empty($pos_settings['is_pay_term_required']); @endphp
		              {!! Form::label('pay_term_number', __('contact.pay_term') . ':') !!} @show_tooltip(__('tooltip.pay_term'))
		              <br/>
		              {!! Form::number('pay_term_number', $transaction->pay_term_number, ['class' => 'form-control width-40 pull-left', 'placeholder' => __('contact.pay_term'), 'required' => $is_pay_term_required]); !!}
		              {!! Form::select('pay_term_type', ['months' => __('lang_v1.months'), 'days' => __('lang_v1.days')], $transaction->pay_term_type, ['class' => 'form-control width-60 pull-left','placeholder' => __('messages.please_select'), 'required' => $is_pay_term_required]); !!}
		            </div>
		          </div>
		        </div>

				@if(!empty($commission_agent))
				@php $is_commission_agent_required = !empty($pos_settings['is_commission_agent_required']); @endphp
				<div class="col-sm-3">
					<div class="form-group">
					{!! Form::label('commission_agent', __('lang_v1.commission_agent') . ':') !!}
					{!! Form::select('commission_agent', $commission_agent, $transaction->commission_agent, ['class' => 'form-control select2', 'id' => 'commission_agent', 'required' => $is_commission_agent_required]); !!}
					</div>
				</div>
				@endif

				<div class="@if(!empty($commission_agent)) col-sm-3 @else col-sm-4 @endif">
					<div class="form-group">
						{!! Form::label('transaction_date', __('sale.sale_date') . ':*') !!}
						<div class="input-group">
							<span class="input-group-addon"><i class="fa fa-calendar"></i></span>
							{!! Form::text('transaction_date', $transaction->transaction_date, ['class' => 'form-control', 'readonly', 'required']); !!}
						</div>
					</div>
				</div>

				@php
					// Keep $status consistent with the UI's select value
					if($transaction->status == 'draft' && $transaction->is_quotation == 1){
						$status = 'quotation';
					} else if ($transaction->status == 'draft' && $transaction->sub_status == 'proforma') {
						$status = 'proforma';
					} else {
						$status = $transaction->status;
					}
				@endphp

				@if($transaction->type == 'sales_order')
					<input type="hidden" name="status" id="status" value="{{$transaction->status}}">
				@else
					<div class="@if(!empty($commission_agent)) col-sm-3 @else col-sm-4 @endif">
						<div class="form-group">
							{!! Form::label('status', __('sale.status') . ':*') !!}
							{!! Form::select(
                                'status',
                                $statuses,
                                $status,
                                [
                                    'class' => 'form-control select2',
                                    'placeholder' => __('messages.please_select'),
                                    'required',
                                    'id' => 'status',
                                    // lock the dropdown for everyone when Final
                                    'disabled' => strtolower($status ?? '') === 'final'
                                ]
                            ) !!}
                            @if (strtolower($status ?? '') === 'final')
                                {{-- disabled inputs don’t submit; mirror value to submit --}}
                                <input type="hidden" name="status" id="status_hidden" value="{{ $status }}">
                            @endif
						</div>
					</div>
				@endif

				@if($transaction->status == 'draft')
				<div class="col-sm-3">
					<div class="form-group">
						{!! Form::label('invoice_scheme_id', __('invoice.invoice_scheme') . ':') !!}
						{!! Form::select('invoice_scheme_id', $invoice_schemes, $default_invoice_schemes->id, ['class' => 'form-control select2', 'placeholder' => __('messages.please_select')]); !!}
					</div>
				</div>
				@endif

                {{-- Invoice number editable for Gate OR Admin OR Accounting --}}
				@if( (method_exists($user,'can') && $user->can('edit_invoice_number')) || $is_admin_role || $is_accounting_role )
				<div class="col-sm-3">
					<div class="form-group">
						{!! Form::label('invoice_no', $transaction->type == 'sales_order' ? __('restaurant.order_no'): __('sale.invoice_no') . ':') !!}
						{!! Form::text('invoice_no', $transaction->invoice_no, ['class' => 'form-control', 'placeholder' => $transaction->type == 'sales_order' ? __('restaurant.order_no'): __('sale.invoice_no')]); !!}
					</div>
				</div>
				@endif

				@php
			        $custom_field_1_label = $custom_labels['sell']['custom_field_1'] ?? '';
			        $is_custom_field_1_required = !empty($custom_labels['sell']['is_custom_field_1_required']) && $custom_labels['sell']['is_custom_field_1_required'] == 1;
			        $custom_field_2_label = $custom_labels['sell']['custom_field_2'] ?? '';
			        $is_custom_field_2_required = !empty($custom_labels['sell']['is_custom_field_2_required']) && $custom_labels['sell']['is_custom_field_2_required'] == 1;
			        $custom_field_3_label = $custom_labels['sell']['custom_field_3'] ?? '';
			        $is_custom_field_3_required = !empty($custom_labels['sell']['is_custom_field_3_required']) && $custom_labels['sell']['is_custom_field_3_required'] == 1;
			        $custom_field_4_label = $custom_labels['sell']['custom_field_4'] ?? '';
			        $is_custom_field_4_required = !empty($custom_labels['sell']['is_custom_field_4_required']) && $custom_labels['sell']['is_custom_field_4_required'] == 1;
		        @endphp

		        @if(!empty($custom_field_1_label))
		        	@php $label_1 = $custom_field_1_label . ($is_custom_field_1_required ? ':*' : ':'); @endphp
		        	<div class="col-md-4">
				        <div class="form-group">
				            {!! Form::label('custom_field_1', $label_1 ) !!}
				            {!! Form::text('custom_field_1', $transaction->custom_field_1, ['class' => 'form-control','placeholder' => $custom_field_1_label, 'required' => $is_custom_field_1_required]); !!}
				        </div>
				    </div>
		        @endif
		        @if(!empty($custom_field_2_label))
		        	@php $label_2 = $custom_field_2_label . ($is_custom_field_2_required ? ':*' : ':'); @endphp
		        	<div class="col-md-4">
				        <div class="form-group">
				            {!! Form::label('custom_field_2', $label_2 ) !!}
				            {!! Form::text('custom_field_2', $transaction->custom_field_2, ['class' => 'form-control','placeholder' => $custom_field_2_label, 'required' => $is_custom_field_2_required]); !!}
				        </div>
				    </div>
		        @endif
		        @if(!empty($custom_field_3_label))
		        	@php $label_3 = $custom_field_3_label . ($is_custom_field_3_required ? ':*' : ':'); @endphp
		        	<div class="col-md-4">
				        <div class="form-group">
				            {!! Form::label('custom_field_3', $label_3 ) !!}
				            {!! Form::text('custom_field_3', $transaction->custom_field_3, ['class' => 'form-control','placeholder' => $custom_field_3_label, 'required' => $is_custom_field_3_required]); !!}
				        </div>
				    </div>
		        @endif
		        @if(!empty($custom_field_4_label))
		        	@php $label_4 = $custom_field_4_label . ($is_custom_field_4_required ? ':*' : ':'); @endphp
		        	<div class="col-md-4">
				        <div class="form-group">
				            {!! Form::label('custom_field_4', $label_4 ) !!}
				            {!! Form::text('custom_field_4', $transaction->custom_field_4, ['class' => 'form-control','placeholder' => $custom_field_4_label, 'required' => $is_custom_field_4_required]); !!}
				        </div>
				    </div>
		        @endif

		        <div class="col-sm-3">
	                <div class="form-group">
	                    {!! Form::label('upload_document', __('purchase.attach_document') . ':') !!}
	                    {!! Form::file('sell_document', ['id' => 'upload_document', 'accept' => implode(',', array_keys(config('constants.document_upload_mimes_types')))]); !!}
	                    <p class="help-block">@lang('purchase.max_file_size', ['size' => (config('constants.document_size_limit') / 1000000)]) @includeIf('components.document_help_text')</p>
	                </div>
	            </div>

		        <div class="clearfix"></div>

		        @if((!empty($pos_settings['enable_sales_order']) && $transaction->type != 'sales_order') || $is_order_request_enabled)
					<div class="col-sm-3">
						<div class="form-group">
							{!! Form::label('sales_order_ids', __('lang_v1.sales_order').':') !!}
							{!! Form::select('sales_order_ids[]', $sales_orders, $transaction->sales_order_ids, ['class' => 'form-control select2 not_loaded', 'multiple', 'id' => 'sales_order_ids']); !!}
						</div>
					</div>
					<div class="clearfix"></div>
				@endif

		        @if(in_array('tables' ,$enabled_modules) || in_array('service_staff' ,$enabled_modules))
		        	<span id="restaurant_module_span" data-transaction_id="{{$transaction->id}}"></span>
		        @endif
			@endcomponent
			
			@component('components.widget', ['class' => 'box-solid'])
				<div class="col-sm-10 col-sm-offset-1">
					<div class="form-group">
						<div class="input-group">
							<div class="input-group-btn">
								<button type="button" class="btn btn-default bg-white btn-flat" data-toggle="modal" data-target="#configure_search_modal" title="{{__('lang_v1.configure_product_search')}}"><i class="fas fa-search-plus"></i></button>
							</div>
							{!! Form::text('search_product', null, ['class' => 'form-control mousetrap', 'id' => 'search_product', 'placeholder' => __('lang_v1.search_product_placeholder'), 'autofocus' => true]); !!}
							<span class="input-group-btn">
								<button type="button" class="btn btn-default bg-white btn-flat pos_add_quick_product" data-href="{{action([\App\Http\Controllers\ProductController::class, 'quickAdd'])}}" data-container=".quick_add_product_modal"><i class="fa fa-plus-circle text-primary fa-lg"></i></button>
							</span>
						</div>
					</div>
				</div>

				<div class="row col-sm-12 pos_product_div" style="min-height: 0">
					<input type="hidden" name="sell_price_tax" id="sell_price_tax" value="{{$business_details->sell_price_tax}}">
					<input type="hidden" id="product_row_count" value="{{count($sell_details)}}">
					@php $hide_tax = session()->get('business.enable_inline_tax') == 0 ? 'hide' : ''; @endphp

					<div class="table-responsive">
					<table class="table table-condensed table-bordered table-striped table-responsive"
						id="pos_table"
						data-require-lot-before-qty="{{ $status_lc === 'final' ? 1 : 0 }}">
						<thead>
							<tr>
								<th class="text-center">@lang('sale.product')</th>
								<th class="text-center">@lang('sale.qty')</th>
								@if(!empty($pos_settings['inline_service_staff']))
									<th class="text-center">@lang('restaurant.service_staff')</th>
								@endif
								<th class="@if(!auth()->user()->can('edit_product_price_from_sale_screen')) hide @endif">@lang('sale.unit_price')</th>
								<th class="@if(!auth()->user()->can('edit_product_discount_from_sale_screen')) hide @endif">@lang('receipt.discount')</th>
								<th class="text-center {{$hide_tax}}">@lang('sale.tax')</th>
								<th class="text-center {{$hide_tax}}">@lang('sale.price_inc_tax')</th>
								@if(!empty($common_settings['enable_product_warranty']))
									<th>@lang('lang_v1.warranty')</th>
								@endif
								<th class="text-center">@lang('sale.subtotal')</th>
								<th class="text-center"><i class="fas fa-times" aria-hidden="true"></i></th>
							</tr>
						</thead>
						<tbody>
							@foreach($sell_details as $sell_line)
								@include('sale_pos.product_row', [
									'product' => $sell_line,
									'row_count' => $loop->index,
									'tax_dropdown' => $taxes,
									'sub_units' => !empty($sell_line->unit_details) ? $sell_line->unit_details : [],
									'action' => 'edit',
									'is_direct_sell' => true,
									'so_line' => $sell_line->so_line,
									'is_sales_order' => $transaction->type == 'sales_order',
									'force_lot_before_qty' => $status_lc === 'final',
								])
							@endforeach
						</tbody>
					</table>
					</div>

					<input type="hidden" name="all_lots_selected" id="all_lots_selected" value="1">

					<div class="table-responsive">
					<table class="table table-condensed table-bordered table-striped table-responsive">
						<tr>
							<td>
								<div class="pull-right">
									<b>@lang('sale.item'):</b> <span class="total_quantity">0</span>
									&nbsp;&nbsp;&nbsp;&nbsp;
									<b>@lang('sale.total'): </b> <span class="price_total">0</span>
								</div>
							</td>
						</tr>
					</table>
					</div>
				</div>
			@endcomponent>

			@component('components.widget', ['class' => 'box-solid'])
				<div class="col-md-4 @if($transaction->type == 'sales_order') hide @endif">
			        <div class="form-group">
			            {!! Form::label('discount_type', __('sale.discount_type') . ':*' ) !!}
			            <div class="input-group">
			                <span class="input-group-addon"><i class="fa fa-info"></i></span>
			                {!! Form::select('discount_type', ['fixed' => __('lang_v1.fixed'), 'percentage' => __('lang_v1.percentage')], $transaction->discount_type , ['class' => 'form-control','placeholder' => __('messages.please_select'), 'required', 'data-default' => 'percentage']); !!}
			            </div>
			        </div>
			    </div>
			    @php
			    	$max_discount = !is_null(auth()->user()->max_sales_discount_percent) ? auth()->user()->max_sales_discount_percent : '';
			    @endphp
			    <div class="col-md-4 @if($transaction->type == 'sales_order') hide @endif">
			        <div class="form-group">
			            {!! Form::label('discount_amount', __('sale.discount_amount') . ':*' ) !!}
			            <div class="input-group">
			                <span class="input-group-addon"><i class="fa fa-info"></i></span>
			                {!! Form::text('discount_amount', @num_format($transaction->discount_amount), ['class' => 'form-control input_number', 'data-default' => $business_details->default_sales_discount, 'data-max-discount' => $max_discount, 'data-max-discount-error_msg' => __('lang_v1.max_discount_error_msg', ['discount' => $max_discount != '' ? @num_format($max_discount) : '']) ]); !!}
			            </div>
			        </div>
			    </div>
			    <div class="col-md-4 @if($transaction->type == 'sales_order') hide @endif"><br>
			    	<b>@lang( 'sale.discount_amount' ):</b>(-) 
					<span class="display_currency" id="total_discount">0</span>
			    </div>
			    <div class="clearfix"></div>

			    <div class="col-md-12 well well-sm bg-light-gray @if(session('business.enable_rp') != 1 || $transaction->type == 'sales_order') hide @endif">
			    	<input type="hidden" name="rp_redeemed" id="rp_redeemed" value="{{$transaction->rp_redeemed}}">
			    	<input type="hidden" name="rp_redeemed_amount" id="rp_redeemed_amount" value="{{$transaction->rp_redeemed_amount}}">
			    	<div class="col-md-12"><h4>{{session('business.rp_name')}}</h4></div>
			    	<div class="col-md-4">
				        <div class="form-group">
				            {!! Form::label('rp_redeemed_modal', __('lang_v1.redeemed') . ':' ) !!}
				            <div class="input-group">
				                <span class="input-group-addon"><i class="fa fa-gift"></i></span>
				                {!! Form::number('rp_redeemed_modal', $transaction->rp_redeemed, ['class' => 'form-control direct_sell_rp_input', 'data-amount_per_unit_point' => session('business.redeem_amount_per_unit_rp'), 'min' => 0, 'data-max_points' => !empty($redeem_details['points']) ? $redeem_details['points'] : 0, 'data-min_order_total' => session('business.min_order_total_for_redeem') ]); !!}
				                <input type="hidden" id="rp_name" value="{{session('business.rp_name')}}">
				            </div>
				        </div>
				    </div>
				    <div class="col-md-4">
				    	<p><strong>@lang('lang_v1.available'):</strong> <span id="available_rp">{{$redeem_details['points'] ?? 0}}</span></p>
				    </div>
				    <div class="col-md-4">
				    	<p><strong>@lang('lang_v1.redeemed_amount'):</strong> (-)<span id="rp_redeemed_amount_text">{{@num_format($transaction->rp_redeemed_amount)}}</span></p>
				    </div>
			    </div>

			    <div class="clearfix"></div>

			    <div class="col-md-4 @if($transaction->type == 'sales_order') hide @endif">
			    	<div class="form-group">
			            {!! Form::label('tax_rate_id', __('sale.order_tax') . ':*' ) !!}
			            <div class="input-group">
			                <span class="input-group-addon"><i class="fa fa-info"></i></span>
			                {!! Form::select('tax_rate_id', $taxes['tax_rates'], $transaction->tax_id, ['placeholder' => __('messages.please_select'), 'class' => 'form-control', 'data-default'=> $business_details->default_sales_tax], $taxes['attributes']); !!}
							<input type="hidden" name="tax_calculation_amount" id="tax_calculation_amount" value="{{@num_format($transaction->tax?->amount)}}" data-default="{{$business_details->tax_calculation_amount}}">
			            </div>
			        </div>
			    </div>
			    <div class="col-md-4 col-md-offset-4 @if($transaction->type == 'sales_order') hide @endif">
			    	<b>@lang( 'sale.order_tax' ):</b>(+) 
					<span class="display_currency" id="order_tax">{{$transaction->tax_amount}}</span>
			    </div>
			    <div class="col-md-12">
			    	<div class="form-group">
						{!! Form::label('sell_note',__('sale.sell_note') . ':') !!}
						{!! Form::textarea('sale_note', $transaction->additional_notes, ['class' => 'form-control', 'rows' => 3]); !!}
					</div>
			    </div>
			    <input type="hidden" name="is_direct_sale" value="1">
			@endcomponent

			@component('components.widget', ['class' => 'box-solid'])
			<div class="col-md-4">
				<div class="form-group">
		            {!! Form::label('shipping_details', __('sale.shipping_details')) !!}
		            {!! Form::textarea('shipping_details',$transaction->shipping_details, ['class' => 'form-control','placeholder' => __('sale.shipping_details') ,'rows' => '3', 'cols'=>'30']); !!}
		        </div>
			</div>
			<div class="col-md-4">
				<div class="form-group">
		            {!! Form::label('shipping_address', __('lang_v1.shipping_address')) !!}
		            {!! Form::textarea('shipping_address', $transaction->shipping_address, ['class' => 'form-control','placeholder' => __('lang_v1.shipping_address') ,'rows' => '3', 'cols'=>'30']); !!}
		        </div>
			</div>
			<div class="col-md-4">
				<div class="form-group">
					{!!Form::label('shipping_charges', __('sale.shipping_charges'))!!}
					<div class="input-group">
					<span class="input-group-addon"><i class="fa fa-info"></i></span>
					{!!Form::text('shipping_charges',@num_format($transaction->shipping_charges),['class'=>'form-control input_number','placeholder'=> __('sale.shipping_charges')]);!!}
					</div>
				</div>
			</div>
			<div class="clearfix"></div>
			<div class="col-md-4">
				<div class="form-group">
		            {!! Form::label('shipping_status', __('lang_v1.shipping_status')) !!}
		            {!! Form::select('shipping_status',$shipping_statuses, $transaction->shipping_status, ['class' => 'form-control','placeholder' => __('messages.please_select')]); !!}
		        </div>
			</div>
			<div class="col-md-4">
		        <div class="form-group">
		            {!! Form::label('delivered_to', __('lang_v1.delivered_to') . ':' ) !!}
		            {!! Form::text('delivered_to', $transaction->delivered_to, ['class' => 'form-control','placeholder' => __('lang_v1.delivered_to')]); !!}
		        </div>
		    </div>
			<div class="col-md-4">
				<div class="form-group">
					{!! Form::label('delivery_person', __('lang_v1.delivery_person') . ':' ) !!}
					{!! Form::select('delivery_person', $users, $transaction->delivery_person, ['class' => 'form-control select2','placeholder' => __('messages.please_select')]); !!}
				</div>
			</div>

            @php
		        $shipping_custom_label_1 = $custom_labels['shipping']['custom_field_1'] ?? '';
		        $is_shipping_custom_field_1_required = !empty($custom_labels['shipping']['is_custom_field_1_required']) && $custom_labels['shipping']['is_custom_field_1_required'] == 1;
		        $shipping_custom_label_2 = $custom_labels['shipping']['custom_field_2'] ?? '';
		        $is_shipping_custom_field_2_required = !empty($custom_labels['shipping']['is_custom_field_2_required']) && $custom_labels['shipping']['is_custom_field_2_required'] == 1;
		        $shipping_custom_label_3 = $custom_labels['shipping']['custom_field_3'] ?? '';
		        $is_shipping_custom_field_3_required = !empty($custom_labels['shipping']['is_custom_field_3_required']) && $custom_labels['shipping']['is_custom_field_3_required'] == 1;
		        $shipping_custom_label_4 = $custom_labels['shipping']['custom_field_4'] ?? '';
		        $is_shipping_custom_field_4_required = !empty($custom_labels['shipping']['is_custom_field_4_required']) && $custom_labels['shipping']['is_custom_field_4_required'] == 1;
		        $shipping_custom_label_5 = $custom_labels['shipping']['custom_field_5'] ?? '';
		        $is_shipping_custom_field_5_required = !empty($custom_labels['shipping']['is_custom_field_5_required']) && $custom_labels['shipping']['is_custom_field_5_required'] == 1;
	        @endphp

	        @if(!empty($shipping_custom_label_1))
	        	@php $label_1 = $shipping_custom_label_1 . ($is_shipping_custom_field_1_required ? ':*' : ':'); @endphp
	        	<div class="col-md-4">
			        <div class="form-group">
			            {!! Form::label('shipping_custom_field_1', $label_1 ) !!}
			            {!! Form::text('shipping_custom_field_1', $transaction->shipping_custom_field_1 ?? null, ['class' => 'form-control','placeholder' => $shipping_custom_label_1, 'required' => $is_shipping_custom_field_1_required]); !!}
			        </div>
			    </div>
	        @endif
	        @if(!empty($shipping_custom_label_2))
	        	@php $label_2 = $shipping_custom_label_2 . ($is_shipping_custom_field_2_required ? ':*' : ':'); @endphp
	        	<div class="col-md-4">
			        <div class="form-group">
			            {!! Form::label('shipping_custom_field_2', $label_2 ) !!}
			            {!! Form::text('shipping_custom_field_2', $transaction->shipping_custom_field_2 ?? null, ['class' => 'form-control','placeholder' => $shipping_custom_label_2, 'required' => $is_shipping_custom_field_2_required]); !!}
			        </div>
			    </div>
	        @endif
	        @if(!empty($shipping_custom_label_3))
	        	@php $label_3 = $shipping_custom_label_3 . ($is_shipping_custom_field_3_required ? ':*' : ':'); @endphp
	        	<div class="col-md-4">
			        <div class="form-group">
			            {!! Form::label('shipping_custom_field_3', $label_3 ) !!}
			            {!! Form::text('shipping_custom_field_3', $transaction->shipping_custom_field_3 ?? null, ['class' => 'form-control','placeholder' => $shipping_custom_label_3, 'required' => $is_shipping_custom_field_3_required]); !!}
			        </div>
			    </div>
	        @endif
	        @if(!empty($shipping_custom_label_4))
	        	@php $label_4 = $shipping_custom_label_4 . ($is_shipping_custom_field_4_required ? ':*' : ':'); @endphp
	        	<div class="col-md-4">
			        <div class="form-group">
			            {!! Form::label('shipping_custom_field_4', $label_4 ) !!}
			            {!! Form::text('shipping_custom_field_4', $transaction->shipping_custom_field_4 ?? null, ['class' => 'form-control','placeholder' => $shipping_custom_label_4, 'required' => $is_shipping_custom_field_4_required]); !!}
			        </div>
			    </div>
	        @endif
	        @if(!empty($shipping_custom_label_5))
	        	@php $label_5 = $shipping_custom_label_5 . ($is_shipping_custom_field_5_required ? ':*' : ':'); @endphp
	        	<div class="col-md-4">
			        <div class="form-group">
			            {!! Form::label('shipping_custom_field_5', $label_5 ) !!}
			            {!! Form::text('shipping_custom_field_5', $transaction->shipping_custom_field_5 ?? null, ['class' => 'form-control','placeholder' => $shipping_custom_label_5, 'required' => $is_shipping_custom_field_5_required]); !!}
			        </div>
			    </div>
	        @endif

	        <div class="col-md-4">
                <div class="form-group">
                    {!! Form::label('shipping_documents', __('lang_v1.shipping_documents') . ':') !!}
                    {!! Form::file('shipping_documents[]', ['id' => 'shipping_documents', 'multiple', 'accept' => implode(',', array_keys(config('constants.document_upload_mimes_types')))]); !!}
                    <p class="help-block">
                    	@lang('purchase.max_file_size', ['size' => (config('constants.document_size_limit') / 1000000)])
                    	@includeIf('components.document_help_text')
                    </p>
                    @php $medias = $transaction->media->where('model_media_type', 'shipping_document')->all(); @endphp
                    @include('sell.partials.media_table', ['medias' => $medias, 'delete' => true])
                </div>
            </div>

	        <div class="clearfix"></div>

	        <div class="col-md-12 text-center">
				<button type="button" class="tw-dw-btn tw-dw-btn-primary tw-text-white tw-dw-btn-sm" id="toggle_additional_expense">
                    <i class="fas fa-plus"></i> @lang('lang_v1.add_additional_expenses') <i class="fas fa-chevron-down"></i>
                </button>
			</div>

			<div class="col-md-8 col-md-offset-4" id="additional_expenses_div">
				<table class="table table-condensed">
					<thead>
						<tr>
							<th>@lang('lang_v1.additional_expense_name')</th>
							<th>@lang('sale.amount')</th>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td>{!! Form::text('additional_expense_key_1', $transaction->additional_expense_key_1, ['class' => 'form-control', 'id' => 'additional_expense_key_1']); !!}</td>
							<td>{!! Form::text('additional_expense_value_1', @num_format($transaction->additional_expense_value_1), ['class' => 'form-control input_number', 'id' => 'additional_expense_value_1']); !!}</td>
						</tr>
						<tr>
							<td>{!! Form::text('additional_expense_key_2', $transaction->additional_expense_key_2, ['class' => 'form-control', 'id' => 'additional_expense_key_2']); !!}</td>
							<td>{!! Form::text('additional_expense_value_2', @num_format($transaction->additional_expense_value_2), ['class' => 'form-control input_number', 'id' => 'additional_expense_value_2']); !!}</td>
						</tr>
						<tr>
							<td>{!! Form::text('additional_expense_key_3', $transaction->additional_expense_key_3, ['class' => 'form-control', 'id' => 'additional_expense_key_3']); !!}</td>
							<td>{!! Form::text('additional_expense_value_3', @num_format($transaction->additional_expense_value_3), ['class' => 'form-control input_number', 'id' => 'additional_expense_value_3']); !!}</td>
						</tr>
						<tr>
							<td>{!! Form::text('additional_expense_key_4', $transaction->additional_expense_key_4, ['class' => 'form-control', 'id' => 'additional_expense_key_4']); !!}</td>
							<td>{!! Form::text('additional_expense_value_4', @num_format($transaction->additional_expense_value_4), ['class' => 'form-control input_number', 'id' => 'additional_expense_value_4']); !!}</td>
						</tr>
					</tbody>
				<table>
			</div>

		    <div class="col-md-4 col-md-offset-8">
		    	@if(!empty($pos_settings['amount_rounding_method']) && $pos_settings['amount_rounding_method'] > 0)
		    		<small id="round_off"><br>(@lang('lang_v1.round_off'): <span id="round_off_text">0</span>)</small>
					<br/>
					<input type="hidden" name="round_off_amount" id="round_off_amount" value=0>
				@endif
		    	<div><b>@lang('sale.total_payable'): </b>
					<input type="hidden" name="final_total" id="final_total_input">
					<span id="total_payable">0</span>
				</div>
		    </div>
			@endcomponent>

			@php $is_enabled_download_pdf = config('constants.enable_download_pdf'); @endphp
			@if($is_enabled_download_pdf && $transaction->type != 'sales_order')
            {{-- Allow Accounting to see this just like Admin --}}
            @if($allow_payments_widget)
				@component('components.widget', ['class' => 'box-solid', 'title' => __('purchase.add_payment')])
					<div class="well row">
						<div class="col-md-6">
							<div class="form-group">
								{!! Form::label("prefer_payment_method" , __('lang_v1.prefer_payment_method') . ':') !!}
								@show_tooltip(__('lang_v1.this_will_be_shown_IN_pdf'))
								<div class="input-group">
									<span class="input-group-addon"><i class="fas fa-money-bill-alt"></i></span>
									{!! Form::select("prefer_payment_method", $payment_types, $transaction->prefer_payment_method, ['class' => 'form-control','style' => 'width:100%;']); !!}
								</div>
							</div>
						</div>
						<div class="col-md-6">
							<div class="form-group">
								{!! Form::label("prefer_payment_account" , __('lang_v1.prefer_payment_account') . ':') !!}
								@show_tooltip(__('lang_v1.this_will_be_shown_IN_pdf'))
								<div class="input-group">
									<span class="input-group-addon"><i class="fas fa-money-bill-alt"></i></span>
									{!! Form::select("prefer_payment_account", $accounts, $transaction->prefer_payment_account, ['class' => 'form-control','style' => 'width:100%;']); !!}
								</div>
							</div>
						</div>
					</div>
				@endcomponent
            @endif
			@endif

			@if($transaction->type == 'sell')
            {{-- Allow Accounting to see this just like Admin --}}
            @if($allow_payments_widget)
				@component('components.widget', ['class' => 'box-solid', 'title' => __('purchase.add_payment')])
				<div class="row">
					@foreach($payment_lines as $payment_line)			
						@if($payment_line['is_return'] == 1)
							@php $change_return = $payment_line; @endphp
							@continue
						@endif

						@if(!empty($payment_line['id']))
        					{!! Form::hidden("payment[$loop->index][payment_id]", $payment_line['id']); !!}
        				@endif

						<div class="payment_row" id="payment_rows_div">
							@include('sale_pos.partials.payment_row_form', ['row_index' => $loop->index, 'show_date' => true, 'payment_line' => $payment_line, 'show_denomination' => true])
						</div>
					@endforeach

					<div class="col-md-12">
		        		<hr>
		        		<strong>@lang('lang_v1.change_return'):</strong><br/>
		        		<span class="lead text-bold change_return_span">0</span>
		        		{!! Form::hidden("change_return", $change_return['amount'] ?? 0, ['class' => 'form-control change_return input_number', 'required', 'id' => "change_return"]); !!}
		        		@if(!empty($change_return['id']))
		            		<input type="hidden" name="change_return_id" value="{{$change_return['id']}}">
		            	@endif
					</div>
				</div>

				<div class="row @if(($change_return['amount'] ?? 0) == 0) hide @endif payment_row" id="change_return_payment_data">
					<div class="col-md-4">
						<div class="form-group">
							{!! Form::label("change_return_method" , __('lang_v1.change_return_payment_method') . ':*') !!}
							<div class="input-group">
								<span class="input-group-addon"><i class="fas fa-money-bill-alt"></i></span>
								@php
									$_payment_method = empty($change_return['method']) && array_key_exists('cash', $payment_types) ? 'cash' : ($change_return['method'] ?? 'cash');
									$_payment_types = $payment_types;
									if(isset($_payment_types['advance'])) { unset($_payment_types['advance']); }
								@endphp
								{!! Form::select("payment[change_return][method]", $_payment_types, $_payment_method, ['class' => 'form-control col-md-12 payment_types_dropdown', 'id' => 'change_return_method', 'style' => 'width:100%;']); !!}
							</div>
						</div>
					</div>
					@if(!empty($accounts))
					<div class="col-md-4">
						<div class="form-group">
							{!! Form::label("change_return_account" , __('lang_v1.change_return_payment_account') . ':') !!}
							<div class="input-group">
								<span class="input-group-addon"><i class="fas fa-money-bill-alt"></i></span>
								{!! Form::select("payment[change_return][account_id]", $accounts, $change_return['account_id'] ?? '' , ['class' => 'form-control select2', 'id' => 'change_return_account', 'style' => 'width:100%;']); !!}
							</div>
						</div>
					</div>
					@endif
					@include('sale_pos.partials.payment_type_details', ['payment_line' => $change_return ?? [], 'row_index' => 'change_return'])
				</div>
				@endcomponent
            @endif
			@endif
			
			
			{{-- Close disabled fieldset BEFORE showing action buttons --}}
                @if($lock_edit)
                    </fieldset>
                @endif

			{{-- ACTION BUTTONS: always visible/clickable --}}
            <div class="row">
              <div class="col-md-12 text-center">
                {!! Form::hidden('is_save_and_print', 0, ['id' => 'is_save_and_print']) !!}
                <button type="button" class="tw-dw-btn tw-dw-btn-primary tw-text-white tw-dw-btn-lg" id="submit-sell">
                  @lang('messages.update')
                </button>
                <button type="button" id="save-and-print" class="tw-dw-btn tw-dw-btn-success tw-text-white tw-dw-btn-lg">
                  @lang('lang_v1.update_and_print')
                </button>
              </div>
            </div>


			@if(in_array('subscription', $enabled_modules))
				@include('sale_pos.partials.recurring_invoice_modal')
			@endif

            {{-- Close the disabled fieldset if we opened it --}}
            @if($lock_edit)
                </fieldset>
            @endif

		</div>
	</div>

	{!! Form::close() !!}
</section>

<div class="modal fade contact_modal" tabindex="-1" role="dialog" aria-labelledby="gridSystemModalLabel">
	@include('contact.create', ['quick_add' => true])
</div>
<div class="modal fade register_details_modal" tabindex="-1" role="dialog" aria-labelledby="gridSystemModalLabel"></div>
<div class="modal fade close_register_modal" tabindex="-1" role="dialog" aria-labelledby="gridSystemModalLabel"></div>
<div class="modal fade quick_add_product_modal" tabindex="-1" role="dialog" aria-labelledby="modalTitle"></div>

@include('sale_pos.partials.configure_search_modal')

@stop

@section('javascript')
  <script src="{{ asset('js/pos.js?v=' . $asset_v) }}"></script>
  <script src="{{ asset('js/product.js?v=' . $asset_v) }}"></script>
  <script src="{{ asset('js/opening_stock.js?v=' . $asset_v) }}"></script>
  @if(in_array('tables' ,$enabled_modules) || in_array('modifiers' ,$enabled_modules) || in_array('service_staff' ,$enabled_modules))
    <script src="{{ asset('js/restaurant.js?v=' . $asset_v) }}"></script>
  @endif

  <script type="text/javascript">
  $(function () {
    const $form   = $('#edit_sell_form');
    const $status    = $('#status');
    const $posTable  = $('#pos_table');

    // ===== keep your original behavior for payments visibility by status =====
    function togglePaymentRows() {
      if ($status.length && $status.val() === 'final') {
        $('#payment_rows_div').removeClass('hide');
      } else {
        $('#payment_rows_div').addClass('hide');
      }
    }
    function syncLotLockMode() {
      if (!$posTable.length) { return; }
      var requireLot = $status.length && $status.val() === 'final';
      $posTable.attr('data-require-lot-before-qty', requireLot ? '1' : '0');
      if (window.BoonPOS_reEvaluateLotLocks) {
        window.BoonPOS_reEvaluateLotLocks();
      }
    }
    togglePaymentRows();
    syncLotLockMode();
    $status.on('change', function(){
      togglePaymentRows();
      syncLotLockMode();
    });

    // ===== File input (kept from your create page for consistency) =====
    $('#shipping_documents').fileinput({
      showUpload: false,
      showPreview: false,
      browseLabel: LANG.file_browse_label,
      removeLabel: LANG.remove
    });

    // ===== Preferred payment account auto-fill (kept) =====
    $(document).on('change', '#prefer_payment_method', function() {
      var default_accounts = $('select#select_location_id').length
        ? $('select#select_location_id').find(':selected').data('default_payment_accounts')
        : $('#location_id').data('default_payment_accounts');

      var payment_type = $(this).val();
      if (payment_type) {
        var default_account = default_accounts && default_accounts[payment_type] && default_accounts[payment_type]['account']
          ? default_accounts[payment_type]['account'] : '';
        var $acct = $('select#prefer_payment_account');
        if ($acct.length && default_accounts) {
          $acct.val(default_account).change();
        }
      }
    });

    // ============================================================
    //  HARD BLOCKS & VALIDATIONS
    //  - Require Lot & Expiry for quotations before qty
    //  - Block save when any row that uses lots has none selected (non-draft)
    //  - OOS / NEGATIVE / ZERO guards
    // ============================================================

    function $qtyInput($row) {
      return $row.find('input.pos_quantity, input.input_quantity, input.quantity, input[name*="[quantity]"]').first();
    }
    function getQty($row) {
      var v = parseFloat($qtyInput($row).val());
      return isNaN(v) ? 0 : v;
    }
    function rowSubtotal($row) {
      var v = 0;
      var $in = $row.find('input.pos_line_total');
      if ($in.length) { v = parseFloat(($in.val()||'').toString().replace(/,/g,'')); }
      if (!v || isNaN(v)) {
        var t = ($row.find('span.pos_line_total_text').text()||'').replace(/[^0-9.\-]/g,'');
        v = parseFloat(t);
      }
      return isNaN(v) ? 0 : v;
    }
    function rowHasPositiveAmount($row) {
      var q = getQty($row);
      if (q > 0) return true;
      var up = parseFloat(($row.find('input.pos_unit_price_inc_tax').val()||'').toString().replace(/,/g,''));
      if (!isNaN(up) && up > 0 && rowSubtotal($row) > 0) return true;
      return rowSubtotal($row) > 0;
    }
    function setQty($row, val) {
      var $q = $qtyInput($row);
      $q.val(val).trigger('change');
    }
    var LOT_SELECTOR = 'select.lot_number, select.lot_no, select.lot_number_dropdown, select[name*="[lot_no_line_id]"], select[name*="[lot_no]"], select[name*="[lot_numbers]"]';
    function lotSelect($row) {
      return $row.find(LOT_SELECTOR).first();
    }
    function requiresLot($row) {
      // If a lot selector exists on the row, treat lot as required
      return lotSelect($row).length > 0;
    }
    function hasLotSelected($row) {
      var $sel = lotSelect($row);
      if (!$sel.length) return true; // not applicable
      var v = $sel.val();
      return v !== undefined && v !== null && (v + '').trim() !== '';
    }
    function isExistingLine($row) {
      if (!$row.length) return false;
      var attr = $row.attr('data-existing-line');
      var data = $row.data('existingLine');
      return attr === '1' || attr === 1 || attr === true || attr === 'true' || data === 1 || data === '1';
    }
    function isQuotationMode() {
      return $status.length && ($status.val() === 'quotation');
    }
    function isDraftMode() {
      return $status.length && ($status.val() === 'draft');
    }
    function lotSelectionEnforced() {
      if ($posTable.length) {
        var attr = $posTable.attr('data-require-lot-before-qty');
        if (typeof attr !== 'undefined') {
          return attr === '1' || attr === 1 || attr === true || attr === 'true';
        }
      }
      return isQuotationMode();
    }
    function getAvailableQty($row) {
      // Prefer currently selected lot/expiry if present
      var $lot = $row.find('select.lot_number, select.lot_no, select.lot_number_dropdown, select[name*="[lot_no_line_id]"], select[name*="[lot_no]"], select[name*="[lot_numbers]"]').first();
      if ($lot.length) {
        var $opt = $lot.find('option:selected');
        if ($opt.length) {
          var lotAvail = parseFloat($opt.data('qty_available'));
          if (isFinite(lotAvail)) {
            return lotAvail;
          }
        }
      }

      // 1) <tr data-qty_available="...">
      var dataAttr = $row.data('qty_available');
      if (dataAttr !== undefined) {
        var n = parseFloat(dataAttr);
        if (!isNaN(n)) return n;
      }
      // 2) common hidden/visible fields or selected lot option text/data
      var candidates = [
        'input.qty_available',
        'input[name*="[qty_available]"]',
        'input[name*="[stock]"]',
        'input[name*="[available_quantity]"]',
        'select.lot_number option:selected'
      ];
      for (var i=0;i<candidates.length;i++){
        var $c = $row.find(candidates[i]).first();
        if ($c.length) {
          var v = parseFloat($c.data('qty_available'));
          if (!isNaN(v)) return v;
          v = parseFloat($c.val() || $c.data('available') || $c.text());
          if (!isNaN(v)) return v;
        }
      }
      // If we cannot detect availability, don't block on edit (assume large)
      return Number.POSITIVE_INFINITY;
    }
    function normalizeQty($row) {
      var q = getQty($row);
      if (!isFinite(q) || q <= 0) {
        setQty($row, 1);
        q = 1;
      }
      var rounded = Math.max(1, Math.floor(q));
      if (rounded !== q) {
        setQty($row, rounded);
      }
      return rounded;
    }
    function updateTotals() {
      $('#pos_table tbody tr.product_row').each(function () {
        var $q = $qtyInput($(this));
        if ($q.length) $q.trigger('change');
      });
    }
    function validateRow($row, showMsg) {
      // Enforce Lot & Expiry selection before positive quantities when required
      var isExisting = isExistingLine($row);
      if (lotSelectionEnforced() && requiresLot($row) && !hasLotSelected($row)) {
        lotSelect($row).addClass('is-invalid');
        var current = getQty($row);
        if (current > 0) {
          if (!isExisting) {
            setQty($row, 0);
          }
          if (showMsg !== false && typeof toastr !== 'undefined') {
            toastr.error((LANG.lot_expiry_required || 'Please select Lot & Expiry first.'));
          } else if (showMsg !== false) {
            alert('Please select Lot & Expiry first.');
          }
        }
        var $ls = lotSelect($row); if ($ls.length) { $ls.focus(); }
        return current > 0 ? false : true;
      } else {
        lotSelect($row).removeClass('is-invalid');
      }

      var avail = getAvailableQty($row);

      // OOS or negative available → remove row and notify
      if (isFinite(avail) && avail <= 0) {
        if (showMsg !== false && typeof toastr !== 'undefined') {
          toastr.error(LANG.out_of_stock || 'Product is out of stock.');
        } else if (showMsg !== false) {
          alert('Product is out of stock.');
        }
        $row.remove();
        updateTotals();
        return false;
      }

      // qty must be >= 1 and integer
      var qty = normalizeQty($row);

      // cap to available if known
      if (isFinite(avail) && qty > avail) {
        if (showMsg !== false && typeof toastr !== 'undefined') {
          var msg = (LANG.quantity_error_msg_in_lot || 'Only {qty} units available.').replace('{qty}', avail);
          toastr.error(msg);
        } else if (showMsg !== false) {
          alert('Only ' + avail + ' units available.');
        }
        setQty($row, avail);
        updateTotals();
      }
      return true;
    }

    // Guard existing rows on load (someone might open an old sale with OOS lines)
    $('#pos_table tbody tr.product_row').each(function(){ validateRow($(this), false); });

    // Guard rows added while editing (search + add)
    var target = document.querySelector('#pos_table tbody');
    if (target) {
      var observer = new MutationObserver(function (mList) {
        mList.forEach(function (m) {
          m.addedNodes.forEach(function (node) {
            var $row = $(node).closest('tr.product_row');
          if ($row.length) {
              // If lot selection is enforced but not selected, force qty 0 immediately
              if (lotSelectionEnforced() && requiresLot($row) && !hasLotSelected($row) && !isExistingLine($row)) {
                setQty($row, 0);
              }
              validateRow($row, true);
            }
          });
        });
      });
      observer.observe(target, { childList: true });
    }

    // Guard qty edits
    $(document).on('input change', '#pos_table tbody input.pos_quantity, #pos_table tbody input.input_quantity, #pos_table tbody input.quantity, #pos_table tbody input[name*="[quantity]"]', function () {
      validateRow($(this).closest('tr.product_row'), true);
    });

    // Lot selection can change available stock
    $(document).on('change', '#pos_table tbody ' + LOT_SELECTOR, function () {
      var $row = $(this).closest('tr.product_row');
      // If lot selection is enforced and qty is 0, bump to 1 upon selection
      if (lotSelectionEnforced() && hasLotSelected($row)) {
        lotSelect($row).removeClass('is-invalid');
        if (getQty($row) <= 0) { setQty($row, 1); }
      }
      validateRow($row, true);
    });

    // Block submit if any invalid remains
    function blockIfInvalid() {
      var ok = true;
      var missingLot = false;
      var firstMissingLotRow = null;
      $('#pos_table tbody tr.product_row').each(function () {
        var $row = $(this);
        if (!validateRow($row, false)) ok = false;
        // Enforce lot selection only when required for this status
        if (lotSelectionEnforced() && requiresLot($row) && !hasLotSelected($row)) {
          missingLot = true;
          lotSelect($row).addClass('is-invalid');
          if (!firstMissingLotRow) firstMissingLotRow = $row;
        }
      });

      var msg = null;
      if (!ok) {
        msg = LANG.out_of_stock || 'One or more items are out of stock.';
      }
      if (missingLot) {
        ok = false;
        msg = (LANG.lot_expiry_required_for_all || LANG.lot_expiry_required || 'Please select Lot & Expiry for all items.');
        if (firstMissingLotRow) {
          try { $('html, body').animate({ scrollTop: firstMissingLotRow.offset().top - 120 }, 200); } catch(e) {}
          var $ls = lotSelect(firstMissingLotRow); if ($ls.length) { $ls.focus(); }
        }
      }
      return { ok: ok, msg: msg };
    }

    $form.on('submit', function (e) {
      var res = blockIfInvalid();
      if (!res.ok) {
        e.preventDefault();
        if (typeof e.stopImmediatePropagation === 'function') { e.stopImmediatePropagation(); }
        var msg = res.msg || (LANG.validation_error || 'Please fix validation errors.');
        if (typeof toastr !== 'undefined') { toastr.error(msg); } else { alert(msg); }
        return false;
      }
    });

    $('#submit-sell, #save-and-print').on('click', function (e) {
      var res = blockIfInvalid();
      if (!res.ok) {
        e.preventDefault();
        if (typeof e.stopImmediatePropagation === 'function') { e.stopImmediatePropagation(); }
        var msg = res.msg || (LANG.validation_error || 'Please fix validation errors.');
        if (typeof toastr !== 'undefined') { toastr.error(msg); } else { alert(msg); }
        return false;
      }
    });
  });
  </script>
@endsection
